/////////////////////////////////////////////////////////////////////////////////

// Original obtained from ShaderToy.com
// Adapted, trivialy, for VGHD by TheEmu.

uniform float u_Elapsed;    // The elapsed time in seconds
uniform vec2  u_WindowSize; // Window dimensions in pixels

// Use defines here rather than edit the body of the code.

#define iGlobalTime u_Elapsed
#define iResolution u_WindowSize
#define iMouse AUTO_MOUSE

/////////////////////////////////////////////////////////////////////////////////

// Simple "Automatic Mouse". Simulates scanning the mouse over the full range of
// the screen with the X and Y scanning frequencies being different. TheEmu.

#define MOUSE_SPEED vec2(0.5,0.577777) * 0.2
#define MOUSE_POS   vec2((1.0+cos(iGlobalTime*MOUSE_SPEED))*u_WindowSize/2.0)
#define MOUSE_PRESS vec2(0.0,0.0)
#define AUTO_MOUSE  vec4( MOUSE_POS, MOUSE_PRESS )

/////////////////////////////////////////////////////////////////////////////////

// The ShaderToy shaders often use textures as inputs named iChannel0. With VGHD
// this may access a Sprite, ClipSprite or ClipNameSprite image depending on how
// the .scn file declares them.
//
// Note, the name used here does not seem to make any difference, so I have used
// iChannel0 which is what is used by ShaderToy but you can use any name as long
// as it matches the use in the main body of the shader. TheEmu.

uniform sampler2D iChannel0;
uniform sampler2D iChannel1;

// With VGHD the range of the P argument's components of the texture functions is
// 0.0 to 1.0 whereas with ShaderToy it seems that the upper limits are given  by
// the number of pixels in each direction, typically 512 or 64.  We therefore use
// the following functions instead.

vec4 texture2D_Fract(sampler2D sampler,vec2 P) {return texture2D(sampler,fract(P));}
vec4 texture2D_Fract(sampler2D sampler,vec2 P, float Bias) {return texture2D(sampler,fract(P),Bias);}

// Rather than edit the body of the original shader we use use a define  here  to
// redirect texture calls to the above functions.

#define texture2D texture2D_Fract

/////////////////////////////////////////////////////////////////////////////////

float distorsion = 0.5 * iMouse.y / iResolution.y;
float radius = 1.5;

const float EPS = 0.000001;
const float MAX = 1000000.0;
const float PI = 3.14159;

float grayscale(vec3 c)
{
    return 0.2126 * c.r + .7152 * c.g + .0722 * c.b;

}

vec2 sphereTexCoords(vec3 normal)
{
  return vec2(fract(atan(normal.x,normal.y) / (2.0*PI) - 0.5 +iGlobalTime*0.025),fract(acos(normal.z) / PI));
}





vec2 intersectSphere( vec3 p, vec3 dir, float r ) 
{
	float b = dot( p, dir );
	float c = dot( p, p ) - r * r;
	
	float d = b * b - c;
	if ( d < 0.0 ) {
		return vec2( -MAX, -MAX );
	}
	d = sqrt( d );
	
	return vec2( -b - d, -b + d );
}


vec2 getIntersection(vec2 dInner, vec2 dOuter)
{
   	float tInner = min(dInner.x,dInner.y);
   	float tOuter = min(dOuter.x,dOuter.y);
    
    if (tOuter < 0.0)
    {
        return vec2(-MAX, -MAX);
    }
    
    if (tInner < 0.0)
    {
       	return vec2(dOuter.x,dOuter.y);
    }
    
    return vec2(tOuter,tInner);
}


float map(vec3 p) 
{   
    vec3 n = normalize(p);
    
    vec2 uv = sphereTexCoords(n);
    
    float v = grayscale( texture2D( iChannel0, uv).rgb);
    
    v *= v * v;
    
    return length(p) - radius + radius * (v - 0.5) * distorsion;
}



vec3 calcNormal( in vec3 pos )
{
	vec3 eps = vec3( 0.01, 0.0, 0.0 );
	vec3 nor = vec3(
	    map(pos+eps.xyy) - map(pos-eps.xyy),
	    map(pos+eps.yxy) - map(pos-eps.yxy),
	    map(pos+eps.yyx) - map(pos-eps.yyx) );
	return normalize(nor);
}


float softshadow( in vec3 ro, in vec3 rd, in float mint, in float tmax )
{
	float res = 1.0;
    float t = mint;
    for( int i=0; i<16; i++ )
    {
		float h = map( ro + rd*t );
        res = min( res, 8.0*h/t );
        t += clamp( h, 0.02, 0.10 );
        if( h<0.001 || t>tmax ) break;
    }
    return clamp( res, 0.0, 1.0 );

}

float calcAO( in vec3 pos, in vec3 nor )
{
	float occ = 0.0;
    float sca = 1.0;
    for( int i=0; i<5; i++ )
    {
        float hr = 0.01 + 0.12*float(i)/4.0;
        vec3 aopos =  nor * hr + pos;
        float dd = map( aopos );
        occ += -(dd-hr)*sca;
        sca *= 0.95;
    }
    return clamp( 1.0 - 3.0*occ, 0.0, 1.0 );    
}

float rand() { return fract(sin(53225.0)*43758.5453123); }


const int STEPS = 128;




void mainImage( out vec4 fragColor, in vec2 fragCoord )
{
    vec2 uv = (-iResolution.xy + 2.0*fragCoord.xy) / iResolution.y;
    
    float innerRadius = radius * (1.0 - distorsion * 0.5);
    float outerRadius = radius * (1.0 + distorsion * 0.5);
    
    
    vec3 origin = vec3(0.0,-4.0,0.0);
    vec3 up = vec3 (0.0, 1.0, 0.0);
    vec3 dir = normalize (vec3 (uv.x, 2.0, uv.y));
    
    vec2 innerSphereDist = intersectSphere(origin,dir,innerRadius);
    vec2 outerSphereDist = intersectSphere(origin,dir,outerRadius);
    
    
    // Calculate t min and max
    vec2 t = getIntersection(innerSphereDist,outerSphereDist);
    /*
    if (t.x < 0.0 || t.y < 0.0) 
    {
            fragColor = vec4(0.0,0.0,0.0,1.0);
            return;
    }*/
    
    vec3 iFront = t.x * dir + origin;
    vec3 iBack = t.y * dir + origin;
    
    vec4 color = vec4(1.0,1.0,1.0,1.0);
    
    dir = normalize(iBack - iFront);
    
    
    float h = length(iBack - iFront) / float(STEPS) / 4.0;
    
    
    
    vec3 pos = iFront + dir * h * (rand() - 0.5) * 4.0;
    
    
    for (int i = 0; i < STEPS; ++i)
    {
        
        float dist = map(pos);
        
        
        if (dist < 0.01 * h) 
        {
         //   fragColor = vec4(1.0,0.0,0.0,1.0);

            break;
        }
        if (i == STEPS -1)
        {
            
            fragColor = vec4(0.0,0.0,0.0,1.0);
            return;
        }
            
        pos += h * dir;
        h *= 1.2;
    }

    
    
     vec3 nor = calcNormal(pos);
      color = texture2D( iChannel0, sphereTexCoords(nor));
    
     vec3 ref = reflect( dir, nor );

    
     // Light position moves with mouse
     
     float theta = iMouse.x / iResolution.x * PI * 2.0;
     vec3 lig = normalize(vec3(cos(theta),sin(theta),0.0));
    
        // lighitng        
        float occ = 0.0 ; //calcAO( pos, nor );
		float amb = clamp( 0.5+0.5*nor.y, 0.0, 1.0 ) * 4.0;
        float dif = clamp( dot( nor, lig ), 0.0, 1.0  );
        float bac = clamp( dot( nor, normalize(vec3(-lig.x,0.0,-lig.z))), 0.0, 1.0 )*clamp( 1.0-pos.y,0.0,1.0);
        float dom = smoothstep( -0.1, 0.1, ref.y );
        float fre = pow( clamp(1.0+dot(nor,dir),0.0,1.0), 2.0 );
		float spe = pow(clamp( dot( ref, lig ), 0.0, 1.0 ),16.0) * grayscale(color.rgb)*4.0;
        
            
            dif *= softshadow( pos, lig, 0.02, 2.5 ) * 1.8;
        //	dom *= softshadow( pos, ref, 0.02, 2.5 );

			vec3 brdf = vec3(0.0);
        	brdf += 1.20*dif*vec3(1.00,0.90,0.60);
			brdf += 1.20*spe*vec3(1.00,0.90,0.60)*dif;
        	brdf += 0.30*amb*vec3(0.50,0.70,1.00)*occ;
        	brdf += 0.40*dom*vec3(0.50,0.70,1.00)*occ;
        	brdf += 0.30*bac*vec3(0.25,0.25,0.25)*occ;
        	brdf += 0.40*fre*vec3(1.00,1.00,1.00)*occ;
			brdf += 0.02;
			color = vec4(color.rgb*brdf,1.0);
            
           
    
    /*
    vec3 iInner = min(innerSphereDist.x,innerSphereDist.y) * dir;
    
    vec3 n_inner = normalize(min(innerSphereDist.x,innerSphereDist.y) * dir + origin);
    vec3 n_outer = normalize(min(outerSphereDist.x,outerSphereDist.y) * dir + origin);
    
    if (innerSphereDist.x > 0.0 || innerSphereDist.y > 0.0)
        color.r = grayscale( texture2D( iChannel0, sphereTexCoords(n_inner)).rgb);
    if (outerSphereDist.x > 0.0 || outerSphereDist.y > 0.0)
        color.g = grayscale( texture2D( iChannel0, sphereTexCoords(n_outer)).rgb);
    */  
    
	fragColor = color;
}

void main ( void )
{
  mainImage ( gl_FragColor, gl_FragCoord.xy );
}
